#
# timing.tcl
#     This is a set of procs to do profiling of Tcl programs.
#     They were taken from an article by Stee Yhler in the Lunux Newsletter.
#
# $Id: timing.tcl,v 1.1.1.1 2004/03/09 15:36:16 clif Exp $
# $Log: timing.tcl,v $
# Revision 1.1.1.1  2004/03/09 15:36:16  clif
# The initial TkReplay from Crowley's website.
#
# Revision 1.2  1996/07/31  16:19:35  crowley
# This is the 1.1 revision of tjreplay.  Lots of small changes.
# Most of them from Dean Brettle.  Connecting and replay are
# faster.  Socket support has been added.
#
# Revision 1.1  1996/07/29  22:48:08  crowley
# Initial revision
#
proc profile_proc {proc} {
    global Template
    rename $proc $proc:OLD
    set body [format $Template $proc]
    proc $proc args $body
}

set Template {
    global Profile__fd
    set time [lindex [time {set result [uplevel [list %1$s:OLD] $args]}] 0]
    set level [expr [info level] - 1]
    if {$level > 0} {
        set caller [lindex [info level $level] 0]
        regsub -all {(.*):OLD} $caller {\1} caller
    } else {
        set caller Toplevel
    }
    catch {puts $Profile__fd [list %1$s $time $caller]}
    return $result
}

proc profile_start {{pattern *}} {
    global Profile__fd
    set Profile__fd [open /tmp/prof.out.[pid] w]
    foreach i [info procs $pattern] {
        profile_proc $i
    }
}

proc profile_stop {} {
    global Profile__fd
    close $Profile__fd
    foreach proc [info procs *:OLD] {
        regsub {(.*):OLD} $proc {\1} restore
        rename $restore {}
        rename $proc $restore
    }
    profile_summarize /tmp/prof.out.[pid]
}

proc profile_calibrate {} {
    global Profile__fd
    proc profile_dummy {args} {return $args}
    set Profile__fd [open /tmp/[pid] w]
    time profile_dummy 10
    set before [lindex [time profile_dummy 10] 0]
    profile_proc profile_dummy
    set after [lindex [time profile_dummy 10] 0]
    close $Profile__fd
    rename profile_dummy {}
    rename profile_dummy:OLD {}
    return [expr ($after - $before)]
}

proc Incr {name {value 1}} {
    upvar $name var
    if {[info exists var]} {
        set var [expr $var + $value]
    } else {
        set var $value
    }
}

proc profile_summarize {file} {
    puts [format "%-20s   calls   ms  ms/call     %%" name]
    set fd [open $file r]
    set total 0
    set overhead [profile_calibrate]
    
    # read in the data, and accumulate the values
    
    while {[gets $fd line] > 0} {
        set name [lindex $line 0]
        set time [lindex $line 1]
        set parent [lindex $line 2]
        Incr count($name)
        Incr sum($name) $time

        if {$parent != "Toplevel"} {
            Incr sum($parent) "- ($time + $overhead)"
        } else {
            Incr total $time
        }
    }
    close $fd
    
    # sort and print the results
    
    foreach name [lsort [array names count]] {
        if {$count($name) == 0} continue
        set ms [expr $sum($name)/1000]
        puts [format "%-20s %4d %7d %8d %8.1f%%" \
            $name $count($name) $ms [expr $ms / $count($name)] \
            [expr $sum($name) * 100.0 / $total]]
    }
}
